// Script to add more past period completions to existing tasks
const { PrismaClient } = require("@prisma/client");
const prisma = new PrismaClient();

async function addMorePastPeriods() {
  console.log("🌱 Adding more past period completions...");

  try {
    // Get all recurring tasks
    const tasks = await prisma.task.findMany({
      where: {
        isRecurring: true,
        recurrencePattern: { in: ["daily", "weekly", "monthly"] },
      },
      include: {
        assignedTo: {
          select: { id: true },
        },
      },
    });

    if (tasks.length === 0) {
      console.log("❌ No recurring tasks found");
      return;
    }

    const today = new Date();
    today.setHours(0, 0, 0, 0);

    for (const task of tasks) {
      const startDate = task.startDate || task.createdAt;
      const daysSinceStart = Math.floor(
        (today - startDate) / (1000 * 60 * 60 * 24)
      );

      if (daysSinceStart <= 0) {
        console.log(
          `⏭️  Skipping ${task.title} - start date is not in the past`
        );
        continue;
      }

      console.log(`\n📅 Processing: ${task.title}`);
      console.log(`   Start Date: ${startDate.toLocaleDateString()}`);
      console.log(`   Days since start: ${daysSinceStart}`);
      console.log(`   Pattern: ${task.recurrencePattern}`);

      let addedCount = 0;
      let skippedCount = 0;

      if (task.recurrencePattern === "daily") {
        // Add completions for past 30 days
        for (let i = 30; i >= 1; i--) {
          const periodDate = new Date(today);
          periodDate.setDate(periodDate.getDate() - i);
          periodDate.setHours(0, 0, 0, 0);

          // Skip if period is before start date
          if (periodDate < startDate) continue;

          // Complete about 70% of periods
          if (i % 3 !== 0) {
            try {
              await prisma.taskPeriodCompletion.create({
                data: {
                  taskId: task.id,
                  periodDate: periodDate,
                  completedAt: new Date(
                    periodDate.getTime() + 10 * 60 * 60 * 1000
                  ), // 10 AM
                  completedById: task.assignedTo?.id || null,
                },
              });
              addedCount++;
            } catch (error) {
              // Period already exists, skip
              skippedCount++;
            }
          } else {
            skippedCount++;
          }
        }
      } else if (task.recurrencePattern === "weekly") {
        // Add completions for past 12 weeks
        for (let i = 12; i >= 1; i--) {
          const weekDate = new Date(today);
          weekDate.setDate(weekDate.getDate() - i * 7);
          const day = weekDate.getDay();
          const diff = weekDate.getDate() - day + (day === 0 ? -6 : 1);
          const monday = new Date(weekDate.setDate(diff));
          monday.setHours(0, 0, 0, 0);

          // Skip if period is before start date
          if (monday < startDate) continue;

          // Complete about 75% of weeks
          if (i % 4 !== 0) {
            try {
              await prisma.taskPeriodCompletion.create({
                data: {
                  taskId: task.id,
                  periodDate: monday,
                  completedAt: new Date(
                    monday.getTime() + 2 * 24 * 60 * 60 * 1000
                  ), // 2 days later
                  completedById: task.assignedTo?.id || null,
                },
              });
              addedCount++;
            } catch (error) {
              skippedCount++;
            }
          } else {
            skippedCount++;
          }
        }
      } else if (task.recurrencePattern === "monthly") {
        // Add completions for past 6 months
        for (let i = 6; i >= 1; i--) {
          const monthDate = new Date(today);
          monthDate.setMonth(monthDate.getMonth() - i);
          const firstDay = new Date(
            monthDate.getFullYear(),
            monthDate.getMonth(),
            1
          );
          firstDay.setHours(0, 0, 0, 0);

          // Skip if period is before start date
          if (firstDay < startDate) continue;

          // Complete about 80% of months
          if (i % 5 !== 0) {
            try {
              await prisma.taskPeriodCompletion.create({
                data: {
                  taskId: task.id,
                  periodDate: firstDay,
                  completedAt: new Date(
                    firstDay.getTime() + 5 * 24 * 60 * 60 * 1000
                  ), // 5 days later
                  completedById: task.assignedTo?.id || null,
                },
              });
              addedCount++;
            } catch (error) {
              skippedCount++;
            }
          } else {
            skippedCount++;
          }
        }
      }

      console.log(`   ✅ Added ${addedCount} completions`);
      console.log(
        `   ⏭️  Skipped ${skippedCount} (already exist or incomplete)`
      );
    }

    console.log("\n✨ Done! Past periods added to all recurring tasks.");
  } catch (error) {
    console.error("❌ Error:", error);
    throw error;
  }
}

addMorePastPeriods()
  .catch((e) => {
    console.error("❌ Failed:", e);
    process.exit(1);
  })
  .finally(async () => {
    await prisma.$disconnect();
  });
